<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Constant;
use App\Filters\ProductsFilter;
use App\Helpers\Format\JsTree;
use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\File;
use App\Models\Product;
use Illuminate\Http\Request;

class ProductsController extends BaseController
{

    public function index(): \Illuminate\Contracts\View\View|\Illuminate\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\Foundation\Application
    {

        $products = Product::query()->filter(new ProductsFilter())->paginate();

        return view('admin.products.all', compact('products'));
    }

    public function create(): \Illuminate\Contracts\View\View|\Illuminate\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\Foundation\Application
    {
        $statuses = Constant::getStatusesViewer();
        $categories = JsTree::getDataFor(Category::class, 'parent_id', ['name' => 'entity', 'value' => Constant::SHOP]);
        return view('admin.products.create', compact('statuses', 'categories'));
    }

    public function store(): \Illuminate\Http\RedirectResponse
    {
        $this->validatePost();

        $data = $this->getData();

        $product = Product::query()->create($data);

        if ($product instanceof Product) {
            $product->tags()->sync(request('tags'));
            $product->categories()->sync(request('categories'));
            return redirect()->route('admin.products.all')->with('success', 'ثبت محصول با موفقیت انجام شد');
        }
        return redirect()->back();
    }

    public function edit(Product $product): \Illuminate\Contracts\View\View|\Illuminate\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\Foundation\Application
    {
        $statuses = Constant::getStatusesViewer();
        $selectedDataIDs = $product->categories()->get(['category_id as id'])->pluck('id')->toArray();
        $categories = JsTree::getDataFor(Category::class, 'parent_id', ['name' => 'entity', 'value' => Constant::SHOP], $selectedDataIDs);
        return view('admin.products.edit', compact('product', 'statuses', 'categories'));
    }

    public function update(Product $product): \Illuminate\Http\RedirectResponse
    {
        $this->validateupdateform($product);
        $data = $this->getData();
        $result = $product->update($data);
        if ($result) {
            $product->tags()->sync(request('tags'));
            $product->categories()->sync(request('categories'));
            return redirect()->route('admin.products.edit', $product)->with('success', 'محصول شما با موفقیت ویرایش شد');
        }
        return redirect()->back();
    }

    public function delete(Product $product): \Illuminate\Http\RedirectResponse
    {
        $product->delete();
        return redirect()->back()->with('delete', 'محصول شما با موفقیت حدف شد');
    }
    public function comments(Product $product): \Illuminate\Contracts\View\View|\Illuminate\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\Foundation\Application
    {
        $comments = $product->comments;

        return view('admin.products.comments', compact('product', 'comments'));
    }

    public function gallery(Product $product): \Illuminate\Contracts\View\View|\Illuminate\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\Foundation\Application
    {
        $files = $product->files;
        return view('admin.products.gallery', compact('product', 'files'));
    }

    public function uploadGallery(Request $request, Product $product): void
    {
        $type = File::getFileType(request()->file('file'));
        $file = new File;
        $file->name = $this->uploadFile(request()->file('file'), Constant::PRODUCT_GALLERY_PATH);
        $file->type = $type;
        $result = $product->files()->saveMany([$file]);

    }

    public function deleteGallery(File $gallery): \Illuminate\Http\RedirectResponse
    {
        $gallery->delete();
        return redirect()->back()->with('delete', 'عملیات شما با موفقیت انجام شد');
    }
    private function validatePost(): void
    {
        request()->validate([

            'description' => ['required'],
            'image' => ['required', 'image'],
            'tags' => ['required'],
            'categories' => ['required'],
            'title' => ['required'],

        ], [

            'description.required' => "توضیحات پست را وارد نمائید",
            'tags.required' => "  تگ پست را وارد نمائید",
            'categories.required' => "  دسته بندی پست را وارد نمائید",
            'image.required' => 'تصویر به درستی بارگزاری نشده است',
            'image.image' => 'تصویر به درستی بارگزاری نشده است',
            'title.required' => 'عنوان را وارد کنید ',

        ]);
    }

    private function validateupdateform(): void
    {
        request()->validate([

            'description' => ['required'],
            'tags' => ['required'],
            'categories' => ['required'],
            'status' => ['required'],
            'title' => ['required'],
        ], [

            'description.required' => "توضیحات پست را وارد نمائید",
            'tags.required' => "  تگ پست را وارد نمائید",
            'categories.required' => "  دسته بندی پست را وارد نمائید",
            'status.required' => "  وضعیت را انتخاب کنید",
            'title.required' => 'عنوان را وارد کنید ',
        ]);
    }

    private function getData(): array
    {
        $data = [
            'description' => request('description'),
            'status' => request('status'),
            'title' => request('title'),
            'short_description' => request('short_description'),
        ];

        if (request()->hasFile('image')) {
            $data['image'] = $this->uploadFile(request()->file('image'), Constant::PRODUCT_AVATAR_PATH);
        }

        return $data;
    }
}
